/**
*
* Copyright 2008 Novatel Wireless Inc.
*
*/

#ifndef _NW_COMMON_API_H
#define _NW_COMMON_API_H

#ifdef __cplusplus
extern "C" {
#endif

/// \addtogroup CoreModule Core Module
///Common and often used functions for managing a WWAN device provided by the \b NvtlSdk component.
/// \n This is the core module used for managing devices.
///
//@{
	/** \file
	This is the main header file for \ref CoreModule.  This SDK header file defines used methods for Core Module.
	*/
	//  SDK function list

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Common API function definitions
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


///
/// Retrieves the SDK version string
///
/// \return
/// type : unsigned short
/// \retval A two byte value indicating an error code.
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// 
/// \param[in]  buffer      A character buffer that will receive the version string
/// \param[in]  buffer_len  The length of the supplied buffer.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetSdkVersion( char* buffer, unsigned short buffer_len );

///
/// Initializes a session in the sdk for use by the client.
///
/// \return
/// type : NOVATEL_API SdkHandle 
/// \retval Returns an SDK handle which should be used for all further interaction with the sdk for this session.
///
/// \remarks This should be the very first SDK method invoked when using the sdk.  This method creates a session context
/// that manages internal state that is referenced during any other sdk method.  This method may be called more than once
/// to create multiple sessions with the SDK, for example to manage multiple devices at once.
NOVATEL_API SdkHandle NOVATEL_CALL NvtlCommon_CreateSession(void);

///
/// Shuts down the sdk session and allows to the SDK to release any resources currently in use.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
///
/// \param[in] session  Handle to the current sdk session
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ReleaseSession( SdkHandle session );

///
///  Configures the debug logging settings for the sdk.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
///
/// \param[in] filename     A complete file path indiciating the name of the log file that should be used
///
/// \param[in] logLevel     The level of logging output desired.
/// \n \b SdkLogLevelNone = 0  No logging output
/// \n \b SdkLogLevelError = 1 Only severe errors are logged
/// \n \b SdkLogLevelWarning = 2 Alerts about non fatal error conditions are logged
/// \n \b SdkLogLevelInfo = 3 General logging output
/// \n \b SdkLogLevelDebug = 4 More detailed debugging information
/// \n \b SdkLogLevelAll = 5 Highest level of logging detail
///
/// \param[in] output       Where logging messages are sent
/// \n \b SdkOutputNone	= 0         No logging output
/// \n \b SdkOutputStdOut = 1       (Windows only) outputs to Message Log
/// \n \b SdkOutputFile	= 2         Outputs to the file specified by the filename parameter
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ConfigureLogging( const char* filename, SdkLogLevelType logLevel, SdkLogOutputType output );


///
///  Configures the maximum file size allowed when logging to file using NvtlCommon_ConfigureLogging().
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
///
/// \param[in] filesize     The maximum size in bytes of the log file, 0 for infinite.  Default is 10Mb.
///
/// \remarks This should be called before NvtlCommon_ConfigureLogging().  This method causes the log file to be 
/// truncated when the file size is greater than the limit specified.  A backup file is created each time 
/// this occurs to preserve log continuity.  As a result, at any point there may be filesize * 2 bytes used   
/// by the backup and active log files.  If at any point the logger is unable to write to the disk, both the 
/// backup and active log files will be truncated.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ConfigureLoggingMaxFileSize( unsigned int filesize );

///
/// Queries the sdk for a list all currently available devices.
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
/// \n \b LR_ERROR_BUFFER_TOO_SMALL if the supplied pDev_list array is not large enough to hold the entire list
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
///
/// \param[in]      session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      pDev_list        A pointer to an array of DeviceDetail structures that will receive the available device list information
/// \param[in,out]  dev_list_size    On input the size of the pDev_list array, on output the number of items filled into the array.
///
/// \remarks This method will return LR_ERROR_OPERATION_NOT_ALLOWED if the system is in the process of suspending or resuming.
/// Due to the behavior of different platforms this function will remain unavailable after resume until a device arrival message is
/// received from the OS, or at worst case for up to 20 seconds after resume.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetAvailableDevices( SdkHandle session, DeviceDetail* pDev_list, unsigned long* dev_list_size);

///
///  Instructs the SDK to attach to the specified device.
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
/// \n \b LR_ERROR_PORT_OPEN_FAILED if the port for the device could not be opened
/// \n \b LR_ERROR_PORT_ALREADY_OPEN if the port for the device is already in use
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
///
/// \param[in] session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] pAttachInfo      A DeviceDetail structure describing the device to attach to and any extra configuration options
///
/// \remarks This method will return LR_ERROR_OPERATION_NOT_ALLOWED if the system is in the process of suspending or resuming.
/// Due to the behavior of different platforms this function will remain unavailable after resume until a device arrival message is
/// received from the OS, or at worst case for up to 20 seconds after resume.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_AttachDevice( SdkHandle session, DeviceDetail* pAttachInfo );

///
/// Instructs the SDK to detach from any currently connected device, but maintains a session with the dll.
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_DetachDevice( SdkHandle session );

///
///  Registers a callback function which the SDK uses to notify clients of general events that occur during a session.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] callback     NvtlEventCallback structure containing the callback function that should be used by the SDK
///
/// \remarks The SDK only supports one callback function per session.  
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_RegisterEventCallback( SdkHandle session, NvtlEventCallback callback );

///
///  Removes the callback function from the SDK's notification list.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] callback     A NvtlEventCallback structure containing the callback function that should be removed from the SDK
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_UnregisterEventCallback( SdkHandle session, NvtlEventCallback callback );

//General API functions
///
///  Returns a structure containing general information regarding the currently attached device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] pDevInfo    A DeviceInfoStruct that will receive the device information.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetDeviceInfo( SdkHandle session, DeviceInfoStruct* pDevInfo );

///
///  Fills a structure containg general information about the network state of the currently attached device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session             Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] pNetworkInfo        A NetworkInfoStruct that will receive the network information.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetNetworkInfo( SdkHandle session, NetworkInfoStruct* pNetworkInfo );

///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// Resets the device paramters to factory default settings.
/// \n A valid Service Programming Code (SPC) is needed for succesful operation.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] spcCode      A string containing the service programming code.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_RestoreToFactoryDefaults( SdkHandle session, char* spcCode );

///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// Resets the device paramters to factory default settings.
/// This version of the call does not reset the device or change power state of the device in any way.
/// A Proper SPC code must be supplied prior to using this method
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_RestoreToFactoryDefaultsBasic( SdkHandle session );

///
///  Checks whether Low Power Mode (LPM) is supported by the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] supported   A pointer to a byte value that will receive the response.  0 = FALSE, 1 = TRUE
///
/// \remarks LPM is supported by Mini PCI and Express devices only.  It is not supported by PCMCIA devices.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_IsLPMSupported( SdkHandle session, unsigned char* supported );

///
/// Returns the current temperature of the device in celcius units.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] temp        A pointer to a two byte number that will receive the temperature.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetTemperature( SdkHandle session, unsigned short* temp );

///
///  This method obtains the nonresettable cumulative call timer from the device. 
/// \n This is the total time that the device has been in a call. 
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] count       A pointer to a 4 byte long value taht receives the total call time in minutes.
///
/// \remarks This method is supported on EVDO devices with \b Sprint related firmware only.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetLifetimeTimer( SdkHandle session, unsigned long* count );

///
///  This method obtains the Network Service speed. 
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  nTheoretical    A 1 byte value that specifies whether the theoretical value is being requested or not.
/// \param[out] speed           A pointer to a value that receives the network service speed.
///
/// \remarks The following values are returned based on the Network Service Type (please refer to method NvtlCommon_GetDeviceNetwork):
/// \n <TABLE border="1" summary="This table lists the values returned by GetDisplaySpeed method.">
/// <CAPTION><EM>Values returned by GetDisplaySpeed method</EM></CAPTION>
/// <TR><TH>Network Service Type<TH>Theoretical Speed<TH>Realistic Speed
/// <TR><TD>NW_SERVICE_GPRS<TD>53.6 Kbps<TD>53.6 Kbps
/// <TR><TD>NW_SERVICE_EDGE<TD>237 Kbps<TD>237 Kbps
/// <TR><TD>NW_SERVICE_UMTS<TD>384 Kbps<TD>384 Kbps
/// <TR><TD>NW_SERVICE_HSDPA<TD>1.8 Mbps<TD>1.8 Mbps
/// <TR><TD>NW_SERVICE_1XRTT<TD>153.6 Kbps<TD>144 Kbps
/// <TR><TD>NW_SERVICE_EVDO<TD>2.4 Mbps<TD>2.4 Mbps
/// <TR><TD>NW_SERVICE_EVDO_REVA<TD>3.1 Mbps<TD>3.1 Mbps
/// <TR><TD>NW_SERVICE_IS95B<TD>0<TD>0
/// <TR><TD>NW_SERVICE_IS95A<TD>0<TD>0
/// <TR><TD>NW_SERVICE_AMPS<TD>0<TD>0
/// <TR><TD>NW_SERVICE_NONE<TD>0<TD>0
/// </TABLE>
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetDisplaySpeed( SdkHandle session, unsigned char nTheoretical, double* speed );

///
/// This method determines whether Auto-install feature is currently enabled on the device or not. Auto-install feature is used to install any connection manager software stored on the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bSet        Paramter that signifies whether a get or set action is being requested
/// \n \b PROP_GET = 0,
/// \n \b PROP_SET = 1,
/// \param[in,out] bEnabled     A pointer to a buffer that contains or receives the setting.  
///
///	\remarks When Auto-install is enabled the device will initially enumerate as a CD-ROM drive on the system. After "ejecting", the device appears normally, as a modem.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_AutoInstallEnabled( SdkHandle session, PropertyAction bSet, unsigned char* bEnabled );

///
/// This method queries or sets the enabled status of the internal (read-only) USB mass storage feature if supported by the device.
/// Enabling or disabling this setting will require a reset of the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bSet        Paramter that signifies whether a get or set action is being requested
/// \n \b PROP_GET = 0,
/// \n \b PROP_SET = 1,
/// \param[in,out] bEnabled     A pointer to a buffer that contains or receives the setting.  
///
/// /remarks If the device also supports a MicroSD slot, then the NvtlEvdo_MicroSDControl method should be used to enable or disable
/// the external MicroSD card.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_InternalMassStorageEnabled( SdkHandle session, PropertyAction bSet, unsigned char* bEnabled );

///
///  Retrieves information pertaining to when the device was last refurbished.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] status          Receives a value that indicates whether the device is refurbished or not, 0 or 1.
/// \param[out] date            A long value that indicates the refurbish date. Following is a hex-format interpretation of this four-byte value: 
/// (MSB...LSB) \b 0Xyyyyddmm
/// \param[out] provider        Receives the name of the provider who refurbished the device.
/// \param[in]  provider_len    The length of the provider buffer.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetRefurbInfo( SdkHandle session, unsigned char* status, unsigned long* date, char* provider, unsigned short provider_len);

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This method sets the preference on network connection and fallback if the preferred network is not available. The typical mode of operation is to connect to an EVDO network, then to fallback to the 1xRTT network when the EVDO
/// network is not available. This functionality can be configured through this method. The device shall need to be reset for changes to take affect. Please refer to SDK method SetPowerMode() in order to reset the device.
/// 
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// 
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  bSet            A Paramter that signifies whether a get or set action is being requested
/// 
/// \param[in] newPrefMode      A value that specifies which network to prefer. Possible values are
///     \n \b SYSTEM_PREFERRED_MODE_AUTO_DETERMINE - 4. The card will automatically determine the best network to attach to.
/// 	\n \b SYSTEM_PREFERRED_MODE_CDMA_ONLY - 9. The card will only attach to CDMA (1xRTT) networks.
/// 	\n \b SYSTEM_PREFERRED_MODE_HDR_ONLY - 10. The card will only attach to HDR (EVDO) networks.
/// 	
/// \param[in] newHybridMode    value that specifies if fallback should be used if the preferred network is not available. Possible values are
/// 	\n \b HYBRID_MODE_NOT_ALLOWED - 0. The card will not fall back to another network if the card is out of coverage of the preferred network.
/// 	\n \b HYBRID_MODE_ALLOWED - 1. The card will fall back to another network if the card is out of coverage of the preferred network.
/// 
/// \param[out] oldPrefMode     A value that specifies the previous value for system / network preference. Returned values are the same as for nNewSystemPreferredMode.
/// \param[out] oldHybridMode   A value that specifies the previous value for hybrid / fallback mode. Returned values are the same as for nNewHybridMode.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_SetSystemPreferredHybridMode( SdkHandle session, PropertyAction bSet, unsigned short newPrefMode, unsigned short newHybridMode, unsigned short* oldPrefMode, unsigned short* oldHybridMode );

//Device Mode functions 
//Probably only Reset is really needed here.
///
///  This method obtains the current power mode for the device. If the device is disabled the method also obtains the cause for it.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session          Handle to the current sdk session returned from NvtlCommon_CreateSession
///
/// \param[out] mode            A value that indicates the current power mode. Possible values are defined by:
///	\n \b DEV_MODE_POWER_OFF = 0
///	\n \b DEV_MODE_FTM = 1
///	\n \b DEV_MODE_OFFLINE = 2
///	\n \b DEV_MODE_OFFLINE_A = 3
///	\n \b DEV_MODE_OFFLINE_D = 4
///	\n \b DEV_MODE_ONLINE = 5
///	\n \b DEV_MODE_LPM = 6
///	\n \b DEV_MODE_RESET = 100
///
/// \param[out] disableMask     A value that indicates whether the device is disabled through hardware or software. This is valid when \e mode indicates DEV_MODE_LPM.
///	Possible values are defined by:
///	\n \b NW_WD_MASK_HW_DISABLE = 0x0001
///	\n \b NW_WD_MASK_SW_DISABLE = 0x0002
///	\n \b NW_WD_MASK_TEMPRATURE_DISABLE = 0x0004
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetPowerMode( SdkHandle session, unsigned short* mode, unsigned char* disableMask );

///
/// This method sets the power mode for the device and can be used to make the device offline or reset it.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] val          A two byte value that specifies the new power mode. Please refer the remarks section for proper usage.
///			Possible values are defined by enum _dev_mode_type:
///			\n \b DEV_MODE_POWER_OFF = 0
///			\n \b DEV_MODE_FTM = 1
///			\n \b DEV_MODE_OFFLINE = 2
///			\n \b DEV_MODE_OFFLINE_A = 3
///			\n \b DEV_MODE_OFFLINE_D = 4
///			\n \b DEV_MODE_ONLINE = 5
///			\n \b DEV_MODE_LPM = 6
///			\n \b DEV_MODE_RESET = 100
///
///			\remarks The only exit from the offline modes is through a restart. Use DEV_MODE_RESET to reset the device thereafter.
///			\n\n When setting the mode to DEV_MODE_ONLINE the mode will be changed to DEV_MODE_RESET if the current mode is offline (DEV_MODE_OFFLINE_A or DEV_MODE_OFFLINE_D).
///			\n\n The following table shows which power modes can be set when. Use the method NvtlCommon_GetPowerMode to determine the current power mode:
///			\n <TABLE border="1" summary="This table states which power modes are valid to be set based on current power mode.">
///					<CAPTION><EM>Setting Power Modes</EM></CAPTION>
///					<TR><TH>SDK Power Mode<TH>Required Power Mode
///					<TR><TD>DEV_MODE_FTM<TD>DEV_MODE_ONLINE
///					<TR><TD>DEV_MODE_OFFLINE_A<TD>DEV_MODE_ONLINE OR DEV_MODE_FTM
///					<TR><TD>DEV_MODE_OFFLINE_D<TD>DEV_MODE_ONLINE OR DEV_MODE_FTM
///					<TR><TD>DEV_MODE_ONLINE<TD>DEV_MODE_FTM OR DEV_MODE_LPM
///					<TR><TD>DEV_MODE_LPM<TD>DEV_MODE_ONLINE OR DEV_MODE_LPM
///					<TR><TD>DEV_MODE_RESET<TD>DEV_MODE_OFFLINE_A OR DEV_MODE_OFFLINE_D
///				</TABLE>
///			\n Resetting a device is a two-step. It requires the device to be offline first. So use SetPowerMode (DEV_MODE_OFFLINE_D) and then use SetPowerMode (DEV_MODE_RESET) 
///			in order to reset the device.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_SetPowerMode( SdkHandle session, unsigned short val );

/// This function is used to enable or disable the WWAN radio.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is unable to enable/disable the radio.
///
/// \param[in]     session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]     bPropAction      A value specifying whether a get or set operation is being requested
/// \param[in,out] state            A short value specifing whether or not the radio is or should be enabled, 0-disabled, 1-enabled
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_RadioState( SdkHandle session, PropertyAction bPropAction, unsigned short* state );


///
/// Forces the device to reset.  This is often required after provisioning or other advanced setup.
/// This will cause the device to power cycle resulting in a sequence of device state events to be fired
/// as the device is rebooted.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Peforms a reset of the deivce using the NvtlCommon_SetPowerMode function.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_Reset( SdkHandle session );

/// 
///	This method sets preferred network mode for use by the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// 
/// \param[in]     session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]     bPropAction      A value indicating whether the property is being set or retrieved.
/// \param[in,out] pref             A byte that contains or receives the prefered network mode.
/// \n For CDMA/EVDO devices use the following values:
/// \n \b NV_PREF_MODE_DIGITAL_PREF = 0
/// \n \b NV_PREF_MODE_DIGITAL_ONLY = 1
/// \n \b NV_PREF_MODE_ANALOG_PREF = 2
/// \n \b NV_PREF_MODE_ANALOG_ONLY = 3
/// \n \b NV_PREF_MODE_AUTOMATIC = 4
/// \n \b NV_PREF_MODE_E911 = 5
/// \n \b NV_PREF_MODE_HOME_ONLY = 6
/// \n \b NV_PREF_MODE_PCS_CDMA_ONLY = 7
/// \n \b NV_PREF_MODE_CELL_CDMA_ONLY = 8
/// \n \b NV_PREF_MODE_CDMA_ONLY = 9
/// \n \b NV_PREF_MODE_HDR_ONLY = 10
/// \n \b NV_PREF_MODE_CDMA_AMPS_ONLY = 11
/// \n \b NV_PREF_MODE_GPS_ONLY = 12
/// \n \b NV_PREF_MODE_GSM_ONLY = 13
/// \n \b NV_PREF_MODE_WCDMA_ONLY = 14
/// \n \n For UMTS and HSPA devices use the following values:
/// \n \b NW_RAT_MODE_AUTO = 0, 
/// \n \b NW_RAT_MODE_GSM = 1, 
/// \n \b NW_RAT_MODE_WCDMA = 2
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_NetworkPreference( SdkHandle session,  PropertyAction bPropAction, unsigned char* pref );

///
/// This method Gets or Sets the NovaSpeed Enabled setting. This setting determines whether the card will optimize network throughput using the NovaSpeed algorithm.
///
/// \return
/// type : unsigned short
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bSet            A value indicating whether the property is being retrieved or set
/// \param[in,out]  NSEnabled       A byte value either containing or receiving the NovaSpeed enabled setting.
/// \n \b 0 - Disable
/// \n \b 1 - Enable
///
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_NovaSpeedEnabled( SdkHandle hClient, PropertyAction bSet, unsigned char* NSEnabled ) ;

//PIN/PUK functions
///
/// This method can be used to determine whether the device is locked or not. If the device is locked the method also obtains information about the kind of lock, 
///	which can be useful while providing the appropriate unlock code in order to unlock the device. This method also retrieves the current autolock setting.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] lockStatus      A long value that specifies the Lock Status. Possible values are defined by:
///	\n \b DEV_UNLOCKED = 0, 
///	\n \b DEV_LOCKED = 1, 
///	\n \b DEV_NETWORK_LOCKED = 2, 
///	\n \b DEV_PIN1_LOCKED = 3, 
///	\n \b DEV_PIN2_LOCKED = 4, 
///	\n \b DEV_PUK1_LOCKED = 5, 
///	\n \b DEV_PUK2_LOCKED = 6, 
///	\n \b DEV_SIM_FAILURE = 7,
///	\n \b DEV_AUTOLOCKED = 0x80
///
/// \param[out] isAutoLockOn    A byte value used as a boolean:
/// \n \b 0 - Auto Lock OFF
///	\n \b 1 - Auto Lock ON
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetLockStatus( SdkHandle session, long* lockStatus, unsigned char* isAutoLockOn );

//PIN/PUK functions
///
/// This method can be used to determine whether the device is locked or not. If the device is locked the method also obtains information about the kind of lock, 
///	which can be useful while providing the appropriate unlock code in order to unlock the device. This method also retrieves the current autolock setting.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] lockStatus      A long value that specifies the Lock Status. Possible values are defined by:
///	\n \b DEV_UNLOCKED = 0, 
///	\n \b DEV_LOCKED = 1, 
///	\n \b DEV_NETWORK_LOCKED = 2, 
///	\n \b DEV_PIN1_LOCKED = 3, 
///	\n \b DEV_PIN2_LOCKED = 4, 
///	\n \b DEV_PUK1_LOCKED = 5, 
///	\n \b DEV_PUK2_LOCKED = 6, 
///	\n \b DEV_SIM_FAILURE = 7,
///	\n \b DEV_AUTOLOCKED = 0x80
///
/// \param[out] isAutoLockOn    A byte value used as a boolean:
/// \n \b 0 - Auto Lock OFF
///	\n \b 1 - Auto Lock ON
///
/// \param[out] retriesLeft     [Optional, GSM Only] A long value that specifies the number of unlock retries left. 
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetLockStatusEx( SdkHandle session, long* lockStatus, unsigned char* isAutoLockOn, long *retriesLeft );

///
/// This method can be used to unlock the device. The correct lock code should be passed in to unlock the device. A locked device cannot make a data connection.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_INVALID_LOCK_CODE if the lock code is invalid
/// \n \b LR_ERROR_DEVICE_ALREADY_UNLOCKED if the device is already unlocked
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] lockType     A byte value that specifies the Lock Type. Possible values are defined as:
/// \n \b DEV_LOCK = 0,
///	\n \b DEV_NET_LOCK = 1,
///	\n \b DEV_PIN1_LOCK = 2,
///	\n \b DEV_PIN2_LOCK = 3,
///	\n \b DEV_PUK1_LOCK = 4,
///	\n \b DEV_PUK2_LOCK = 5,
///	\n \b DEV_AUTOLOCK = 6	
///
/// \param[in] lockCode     A string value that specifies the Lock code
///
/// \remarks It is useful to know the kind of lock on the device so that the appropriate lock type can be specified while unlocking the device.
///	Use the method GetLockStatus() to obtain information about the lock type. For example, if the method GetLockStatus() returns DEV_PIN1_LOCKED 
///	then the lock type DEV_PIN1_LOCK should be specified in the call to Unlock method.
///	\n\n The method GetDeviceState() will return NW_DEVICE_STATE_LOCKED for a device that is locked.
///	\note Care should be taken while using the unlock code with a UMTS/HSPA a device. Generally, there are two PINs; PIN1 and PIN2. PIN1 is the "global" PIN which is the first level of SIM locking.
///	If the SIM is "PIN Locked" then it is PIN1 that is required to gain access to any functionality of the SIM. If, when requested to do so, the user incorrectly enters PIN1 three times the PIN 
///	will be locked out. The user will require a PUK code (PUK1) and the PIN (PIN1) to regain access to the SIM. If the PUK is incorrectly entered 10 times, both the PUK (PUK1) and the PIN (PIN1) 
///	will be locked out rendering the SIM fairly useless.
///	PIN2 and its associated PUK2 work almost the same way except that PIN2 is considered a "local" PIN and, if activated only provides access to a narrow scope of features on the SIM,
///	such as the FDN (Fixed Dialing Number list) or BDN (Barred Dialing Number List) phone books (if they are available and PIN locked). The exception to PIN2 behaving like PIN1 is that 
///	if PUK2 is entered incorrectly 10 times it will not render the SIM useless. Only the features that were PIN2 accessible will remain inaccessible. PIN1 and PIN2 are mutually exclusive.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_UnlockDevice( SdkHandle session, unsigned char lockType, char* lockCode );

///
/// This method can be used to turn ON or OFF the autolock setting. The correct lock code should be passed in to successfully change this setting. 
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_INVALID_LOCK_CODE if the lock code is invalid
/// \b \b LR_ERROR_DEVICE_ALREADY_LOCKED if the device is already locked
///
/// \param[in] session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] lockOn           byte value used as a boolean to set autolock:
///	\n \b 0 - Auto Lock OFF
///	\n \b 1 - Auto Lock ON
///
/// \param[in] lockCode         A string value that specifies the Lock code
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_SetAutoLock( SdkHandle session, unsigned char lockOn, char* lockCode );

///
/// This method can be used to change the lock code of the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_INVALID_LOCK_CODE if the lock code is invalid
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] szOldCode    A string value that specifies the \b Old lock code. For EVDO devices the required length of the lock code is 4, which is also the maximum length. For UMTS/HSPDA devices maximum length of the lock code is 8.
/// \param[in] szNewCode    A string value that specifies the \b New lock code.  For EVDO devices the required length of the lock code is 4, which is also the maximum length. For UMTS/HSPDA devices maximum length of the lock code is 8.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ChangeLockCode( SdkHandle session, char* szOldCode, char* szNewCode );

//Sms functions
///
/// This function checks to see if SMS is supported on the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] bSupported  A one byte boolean value indicating 0 for not supported and 1 for supported.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_IsSmsSupported( SdkHandle hClient, unsigned char* bSupported );



///
/// This function requests a single Sms message to be read from the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  theIndex       The index of the sms message to be read.
/// \param[out] msg         A SmsMessageStruct struct that may receive the sms message
///
/// \remarks For EVDO devices this function returns LR_SMS_READ_MSG_ASYNC_REPSONSE and the sms message is received asynchronously
/// through the event handler with the event NW_EVENT_SMS_READ_BUFFER
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ReadSmsMsg( SdkHandle session, unsigned short theIndex, SmsMessageStruct* msg );

///
/// This function sends an encoded sms message as described by the message struct.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] msg          A SmsMessageStruct struct containing the encoded sms message.
///
/// \remarks For EVDO devices the send status of the operation is received asynchronously through the event handler with the event NW_EVENT_SMS_SEND_STATUS
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_SendSmsMsg( SdkHandle session, SmsMessageStruct* msg );

///
/// This function deletes an sms message from the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] theIndex        The index of the sms message to be deleted
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_DeleteSmsMsg( SdkHandle session, unsigned short theIndex );

///
/// This function retrieves information about the sms messages on the device
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_BUFFER_TOO_SMALL if the supplied info_array is not large enough to hold the entire message list
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  eSmsType    Specifies the type of messages to get information about.  This value can be
/// \n 												NW_SMS_RECEIVED_UNREAD to get a list of newly received messages
/// \n 												NW_SMS_RECEIVED_READ to get a list of alread read messages
/// \n 												NW_SMS_RECEIVED_ALL to get a list of both read and unread messages
/// \param[in]  info_array  An array of SmsMessageInfo structs that will receive the information about the avaialble messages.
/// \param[in]  array_size  The number of elements in the supplied info_array
/// \param[in]  msg_count   The number of elements available or copied into the array
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetSmsList( SdkHandle session, SmsStateType eSmsType, SmsMessageInfo* info_array, unsigned short array_size, unsigned short* msg_count );

///
/// This function returns the local Network time at which an sms message was received on the device.
/// This function is support only for certain CDMA devices with Verizon firmware.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  index       The message index of which the local received timestamp should be returned.
/// \param[out] timestamp   A pointer to an NvtlTimeStruct which will receive the timestamp
///
/// /remarks This function is support only for certain CDMA devices with Verizon firmware.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetSmsRxTimestamp( SdkHandle hClient, unsigned short index, NvtlTimeStruct* timestamp);

///
/// This function erases all of the stored local sms received timestamps.
/// This function is support only for certain CDMA devices with Verizon firmware.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
///
/// /remarks This function is support only for certain CDMA devices with Verizon firmware.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ClearSmsRxTimestamps( SdkHandle hClient );

//Address Book functions
///
///  Reads a single address book contact from the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  theIndex       A value containing the index of the address book entry to read.
/// \param[out] pContact    A ContactInfoStruct structure that will receive the contact information.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ReadContact( SdkHandle session, unsigned short theIndex, ContactInfoStruct* pContact );

///
///  Reads multiple address book contacts from the device. (UMTS/HSPA devices only)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  start_index     A value containing the starting index of the address book to read.
/// \param[in]  num_to_read     Value specifying the size of the ContactInfoStructr array and the number of contacts to try reading
/// \param[out] pContact        Array of ContactInfoStruct structures that will receive the contact information.
/// \param[out] pRead           A value that receives the number of contacts read
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_ReadMultipleContacts( SdkHandle session, unsigned short start_index, unsigned short num_to_read, ContactInfoStruct* pContact, unsigned short* pRead );

///
///  Writes a single contact to the address book on the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] pContact     A ContactInfoStruct structure containing information to be written to the address book on the device.
///
/// \remarks To delete an entry from the address book the a contact should be written with empty values for the name and details
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_WriteContact( SdkHandle session, ContactInfoStruct* pContact );

///
///  This function receives information about the size of the address book and its entries.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_DEVICE_ALREADY_UNLOCKED if the device is UMTS/HSPA and the sim is not yet ready
///
/// \param[in]  session             Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] pContactDetails     A ContactSizeInfoStruct structure that will receive the address book size information.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetContactSizeDetails( SdkHandle session, ContactSizeInfoStruct* pContactDetails );

///
///  This function retrieves the current data sent and recieved counters stored on the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] dataTx      A long value that will receive the data sent counter.
/// \param[out] dataRx      A long value that will receive the data received counter.
///
/// /// \remarks This method is supported on EVDO devices with \b Sprint related firmware only.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetDataCounters( SdkHandle session, unsigned long* dataTx, unsigned long* dataRx );

///
///  <tt><b>\<CDMA/EVDO devices only\></b></tt> \n
///			This method retrieves the state and the charge left on the battery, if present on the device.
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] state       A short value that will receive the state of the battery.
/// \n \b BATTERY_POWERED   - Battery is present; Powered by battery.
/// \n \b BATTERY_EXTERNAL_POWER - Battery is present; Powered by external source.
/// \n \b BATTERY_UNAVAILABLE   - Battery is absent.
/// \param[out] chargePct   A short value that will receive the percentage charge left on the battery.
/// \remarks This method is supported on only on devices that have a batery.
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetBatteryState( SdkHandle session, unsigned short* state, unsigned short* chargePct );

///
/// This function retreives the device manufacturer name
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in,out]  buffer          Buffer that will recieve the manufacturer value
/// \param[in]      buffer_len      Total length of the buffer that will receive the manufacturer
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetManufacturer( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
/// <tt><b>\<Windows 7 Only\></b></tt>\n
/// If a Windows 7 compiliant device and driver are in use, this method returns the interface id that can be used to 
/// identify and control the device using the Windows 7 Mobile Broadband API 
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in,out]  buffer          Buffer that will recieve the Windows7 interface id value
/// \param[in]      buffer_len      Total length of the buffer that will receive the interface id
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetWindows7InterfaceId( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
/// Returns the Media Access Control address (MAC address) for the currently attached device
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in,out]  buffer          Buffer that will recieve the MAC address value
/// \param[in]      buffer_len      Total length of the buffer that will receive the MAC address
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetMACAddress( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
/// Controls whether or not the 3G transmitter is running at full or limited power mode.
///  (RedTail/Wimax device only)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      mode            A value containing the 3G limited power setting.  Possible values are defined as:
///	\n \b 0  Full Power operation
///	\n \b 1  Limited Power operation
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_Set3GLimitedPower( SdkHandle session, U8 mode );

///
/// Notifies 3G FW of the current 4G/Wimax call state.
///  (RedTail/Wimax device only)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      status          A value containing the status of the current 4G/Wimax connection
///	\n \b 0  DEV_UNINITIALIZED
///	\n \b 1  DEV_RF_OFF_HW_SW 
///	\n \b 2 - DEV_RF_OFF_HW 
///	\n \b 3 - DEV_RF_OFF_SW 
///	\n \b 4 - DEV_READY 
///	\n \b 5 - DEV_SCANNING 
///	\n \b 6 - DEV_CONNECTING 
///	\n \b 7 - DEV_DATA_CONNECTED 
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_DeviceStatusNotify4G( SdkHandle session, U8 status);

///
/// Gets or sets the 4G led configuration.
///  (RedTail/Wimax device only)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bPropAction     A value specifying whether a get or set operation is being requested
/// \param[in]      pConfig         A pointer to an LedConfig4G structure that either contains or will receive setting.
/// Color fields are defined as:
///	\n \b 0  Red
///	\n \b 1  Green
///	\n \b 2  Blue
///	\n \b 255  No color
///
/// Blinking on/off fields can be defined as 
/// \n \b 0  65535ms 
///
/// Led Pattern field can be defined as
/// \n \b 0  All Off
/// \n \b 1  On (solid pattern)
/// \n \b 2  Blink
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_LedConfiguration4G( SdkHandle session, PropertyAction bPropAction, LedConfig4G *pConfig);

///
/// Gets or sets the Dual Work Mode setting.
///  (RedTail/Wimax device only)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      work_mode       A pointer to a byte that either contains or will receive the work mode setting.
///	\n \b 0 - Not yet provisioned (factory default)
///	\n \b 1 - CDMA Only
///	\n \b 2 - CDMA & WIMAX 
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_DualWorkModeSetting4G( SdkHandle session, PropertyAction bSet, U8* work_mode );

///
/// Determines whether or not the device is operating in an express card slot or an external caddy.
///  (RedTail/Wimax device only)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      mode            A pointer to a byte that will receive the mode .
///	\n \b 0 - Operating in external caddy
///	\n \b 1 - Operating in Express card slot
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_GetInterfaceMode( SdkHandle session, U8* mode );

///
///  Returns the current network type for Multi-Mode and LTE capable devices.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     eNetworkType    A pointer to a value that will contain one of the enumeration values from NW_LTE_NETWORK_TYPE.
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_GetLTENetworkType( SdkHandle hClient, LTENetworkType *eNetworkType );

///
///  Returns the device's MSISDN value.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     buffer          A pointer to a string that will contain the MSISDN value of size 42.
/// \param[in/out]  len             A pointer to an integer containing the buffer length on input and the 
///                                 required buffer length on output.
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_GetMSISDN( SdkHandle hClient, char *buffer, unsigned short *len );

///
///  This method retrieves or sets the APN String of an APN class type for Multi-Mode and LTE capable devices.
///
/// \return
/// type : unsigned short
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bSet         Paramter that signifies whether a get or set action is being requested
/// \param[in]      apnClass     A pointer to APN class type
/// \param[in,out]  apnString    A pointer to buffer that contains or receives APN String
///
NOVATEL_API U16 NOVATEL_CALL NvtlCommon_LteAPN( SdkHandle session, U8 bSet, U8 *apnClass, U8 *apnString);

///
///  This method retrieves the IP addresses (IPv4 and IPv6) for 4G Multi-Mode devices.
///
/// \return
/// type : unsigned short
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
///
/// \param[in]      session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     IPAddress    Pointer to a MultiModeIPAddress struct containing the current IP address for the device
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlCommon_GetMultiModeIpAddress( SdkHandle session, MultiModeIPAddress* IPAddress );

//END COMMON API FUNCTIONS
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//GSM/HSPA/UMTS functions
///
///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt>\n
/// This function sets or retrieves the domain attach state for the device
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]     session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]     bPropAction      A value specifying whether a get or set operation is being requested
/// \param[in,out] state            A short value specifing whether or not the device is or should attach to the network, 0-detach, 1-attach
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_DomainAttach( SdkHandle session, PropertyAction bPropAction, unsigned short* state );

///  <tt><b>\<UMTS and HSPA devices only\></b></tt>
/// This function sets or retrieves the phone number for the Short Message Service Center (SMSC)
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]     session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]     bPropAction      A value specifying whether a get or set operation is being requested
/// \param[in,out] buffer           When setting, a buffer that contains the SMSC to be set.  When retrieving, a buffer that recieves the current SMSC.
/// \param[in,out] buffer_len       When setting, the length of the SMSC being provided.  When retrieving, the total length of the buffer being provided to the function.
///
/// \remarks The SMSC number is not overwritten if it already exists.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_SMSC( SdkHandle session, PropertyAction bPropAction, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt>\n
/// This function retreives the International Mobile Subscriber Identity (IMSI) of the SIM.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in,out]  buffer          Buffer that will recieve the IMSI value
/// \param[in]      buffer_len      Total length of the buffer that will receive the IMSI
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetImsi( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt>\n
/// This function retreives the Integrated Circuit Card ID (ICCID) of the SIM.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]   session        Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]  buffer         A buffer that will recieve the ICCID value
/// \param[in]   buffer_len     Total length of the buffer that will receive the ICCID vallue
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetIccid( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
/// <tt><b>\<UMTS and HSPA devices only\></b></tt>\n
/// This method obtains the MCC (Mobile Country Code) and MNC (Mobile Network Code) of the network to which the device is currently attached.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]   session        Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]  buffer         A buffer that will receive the MCCMNC value
/// \param[in]   buffer_len     The total length of the buffer that will receive the MCCMNC value
///
/// \remarks The value returend is a string value that indicates the MCC and MNC. For e.g. "310380" is returned for AT&T Wireless, where "310" is MCC for USA and "380" is the MNC for AT&T Wireless.
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetNetworkMCCMNC( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
///	This method obtains the network operator to which the device is currently attached.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
/// \n \b LR_ERROR_BUFFER_TOO_SMALL if the supplied buffer is not large enough to hold the response
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] buffer      A buffer that will receive the network operator value
/// \param[in]  buffer_len  The total length of the buffer that will receive the network operator value
///
///  \remarks This function uses the AT+COPS command to retrieve the current network operator
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetNetworkOperator( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
///	This function obtains a list of operators detected by the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]      session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     buffer      A buffer that will receive the list of network operators.
/// \param[in, out] buffer_len  The total length of the buffer that will receive the network operator list.
///
/// \remarks The AT+COPS command is used to retreive the network operators' list.  This command may take up to 30 seconds or more to scan the operator list.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetNetworkOperatorList( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
///	This function obtains issues a single AT command to the device
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
/// \n \b LR_ERROR_BUFFER_TOO_SMALL if the supplied response buffer is not large enough to hold the entire response
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  command         A string containing the AT command to issue.
/// \param[in]  prompt          Optional.  A string containing a second piece of data to be issued if the AT command being executed waits for further input at a prompt.  Eg.  AT+CMGS
/// \param[out] response        A buffer that will receive the response of the AT command.
/// \param[in]  response_len    The total length of the buffer that will receive the response.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_SendATCommand( SdkHandle session, char* command, char* prompt, char* response, unsigned long response_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
///	This function sets or retrieves the bands currently set on the device.
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bPropAction     A value indicating whether the band is being set or retrieved.
/// \param[in,out]  band            A value containing either the bands to set, or the bands being retrieved.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_Band( SdkHandle session, PropertyAction bPropAction, unsigned long* band );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
/// This function sends an SMS message that is currently stored on the SIM.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  theIndex       The index of the SMS message to send
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_SendStoredSms( SdkHandle session, unsigned short theIndex );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
/// This function sets the storage memory used for SMS messages.  Eg "SM"
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in] session      Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] name         A string containing the name of the storage memory to use
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_SetSmsStorage( SdkHandle session, const char* name );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
/// This function sets the preferred phone book storage area.  Eg, "SM", "FD", "LD", "ON"
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in] session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in] storage_name    A string containing the name of the storage memory to use
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_SetContactStorage( SdkHandle session, char* storage_name );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
/// This function retrieves information about the storage aread currently in use for the phone book.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] numRecords      Recieves the number of records currently in use
/// \param[out] maxRecords      Receives the max number of records allowed
/// \param[out] buffer          The name of the storage area currently in use
/// \param[out] buffer_len      The length of the buffer that receives the storage area name
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetContactStorage( SdkHandle session, unsigned short* numRecords, unsigned short* maxRecords, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
/// This function writes an SMS message to the SIM but does not send it.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  msg         A SmsMessageStruct structure containing the message to be written.
/// \param[out] theIndex       A value that receives the index to where the message was written.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_WriteSmsMsg( SdkHandle session, SmsMessageStruct* msg, unsigned short* theIndex);

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt> \n
/// This function checks to see if all of the storage memory for SMS messages is in use.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] full        A value that receives the status of the SMS memory. 0 = not full.  1 = full.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_IsMsgMemoryFull( SdkHandle session, unsigned char* full );

///
///  <tt><b>\<UMTS and HSPA devices only\></b></tt>\n
/// This method sets the network operator to be used by the device. 
/// The new value that is specified should be provided by the GetNetworkOperatorList function.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  mode        A value that specifies the command mode. Possible values are defined by :
/// \n \b COPS_MODE_AUTOMATIC = 0
/// \n \b COPS_MODE_MANUAL = 1
/// \n \b COPS_MODE_DEREGISTER = 2
/// \n \b COPS_MODE_SET_FORMAT = 3
///
/// \param[in] format       A value that specifies the format of szOper parameter. Possible values are defined by:
///	\n \b COPS_FORMAT_LONG_ALPHANUMERIC = 0, 
///	\n \b COPS_FORMAT_SHORT_ALPHANUMERIC = 1, 
///	\n \b COPS_FORMAT_NUMERIC = 2, 
///
/// \param[out] accessTech      A value that specifies the network access technology to use. T Possible values are defined by:
///	\n \b ACCESS_TECH_GSM			= 0
/// \n \b ACCESS_TECH_GSM_COMPACT	= 1
/// \n \b ACCESS_TECH_UTRAN		= 2
/// \n \b ACCESS_TECH_AUTOMATIC	= 3
/// \n \b ACCESS_TECH_NA			= 4
///
/// \param[in] szOper A string value that specifies the Network Operator to be set. This should be a value returned by GetNetworkOperatorList function.
///
/// /remarks This method is a wrapper around the AT+COPS command.  This function can be used to perform the various operations performed by AT+COPS
/// depending upon the parameters supplied.
/// \n
/// \n \b "Automatic carrier selecton" Supplying a mode value of COPS_MODE_AUTOMATIC will result in an "AT+COPS=0" command being issued.  All other parameters are ignored.
/// \n
/// \n \b "Set +COPS format" Supplying a mode value of COPS_MODE_SET_FORMAT and an appropriate mode value will issue an at command to set the output format.  The operator field should be blank or empty string.
/// \n
/// \n \b "Manual network selection"  Supplying a mode value of COPS_MODE_MANUAL and an appropriate operator string will result in a command to manually attach to a specific network.
/// \n To specify a specific network type in the command supply an accessTech value of ACCESS_TECH_GSM, ACCESS_TECH_COMPACT, or ACCESS_TECH_UTRAN
/// \n If you do not wish to indicate a specific technology type in the command supply ACCESS_TECH_AUTOMATIC.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_SetNetworkOperator( SdkHandle session, unsigned short mode, unsigned short format, unsigned short accessTech, char* szOper);

///
/// <tt><b>\<HSPA devices only\></b></tt>\n
///	This method obtains Enhanced Network Selection (ENS) information.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] ensMode     A value that receives the manual network selection indicator
///	\n \b 0 - Manual Network selection is not supported by network
///	\n \b 1 - Manual Network selection is supported by network
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_GetEnsInfo( SdkHandle session, unsigned long* ensMode );

///
/// <tt><b>\<HSPA devices only\></b></tt>\n
///	Uses AT+COPS to determine if the device was manually assigned to a network.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] bManual     A value that receives the manual network selection indicator
NOVATEL_API unsigned short NOVATEL_CALL NvtlGsm_IsManualNetworkSelected( SdkHandle session, unsigned char* bManual);
//End GSM functions
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//CDMA functions
//General CDMA

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This method gets or sets the current roaming preference on the device.
/// Modifying this setting will require a reset of the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]     session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]     bPropAction      A value indicating whether the property is being set or retrieved.
/// \param[in]     nam              A value that specifies Nam to use
/// \param[in,out] pref             RoamPreferenceType contains or receives the roaming preference. Possible values are defined by:
/// \n \b NV_ROAM_PREF_HOME = 1
/// \n \b NV_ROAM_PREF_AFFIL = 3
/// \n \b NV_ROAM_PREF_ROAM = 6
/// \n \b NV_ROAM_PREF_ANY = 255
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_RoamPreference( SdkHandle session,  PropertyAction bPropAction, unsigned char nam, RoamPreferenceType* pref );

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This method gets or sets whether to use international roaming when roaming is enabled.
/// Modifying this setting will require a reset of the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// \n \b LR_ERROR_API_NOT_SUPPORTED if the device does not support international roaming
/// 
/// \param[in]     session          Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]     bPropAction      A value indicating whether the property is being set or retrieved.
/// \param[in,out] bRoam            A binary value that sets or indicates whether international roaming is on (1) or off (0).
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_IntlRoamPreference( SdkHandle hClient, PropertyAction bSet, unsigned char* bRoam );


/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
///	This method sets the band preference on the device.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]      session             Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bPropAction         A value indicating whether the property is being set or retrieved.
/// \param[in]      nam                 A value that specifies Nam to use
///	\param[in,out]  pref                A BandPreferenceType contains or receives the Band Preference. Possible values are defined by enum _NV_BAND_PREF_E_TYPE:
///	\n \b NV_BAND_PREF_CELL = 3
///	\n \b NV_BAND_PREF_PCS = 4
///	\n \b NV_BAND_PREF_AUTOMATIC = 65535
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_BandPreference( SdkHandle session,  PropertyAction bPropAction, unsigned char nam, BandPreferenceType* pref );

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
///	This method obtains the date and time information as provided by the network to which the device is currently attached.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] ts          A NvtlTimeStruct that receives the time information.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_GetDateTime( SdkHandle session, NvtlTimeStruct* ts );

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
///	This method obtains the Extended Roaming Indicator (ERI) network roaming information from the device. 
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out] eriInfo     A EriInfoStruct that receives the extending roaming information.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_GetEriInfo( SdkHandle session, EriInfoStruct* eriInfo );


/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
///	This method enables the on or off setting of the modem led light.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]      session             Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bPropAction         A value indicating whether the property is being set or retrieved.
/// \param[in, out] enabled             When retrieving the value, a pointer to a byte that receives the current on off state.
///                                     When setting, a pointer to a byte that contains the on or off state to set, 0 or 1.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_LedControl( SdkHandle session, PropertyAction bPropAction, unsigned char* enabled );

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
///	This method controls the micro sd functionality of the modem.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      bPropAction     A value indicating whether the property is being set or retrieved.
/// \param[in, out] enabled         When retrieving the value, a pointer to a byte that receives the current on off state.
///                                 When setting, a pointer to a byte that contains the on or off state to set, 0 or 1.
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_MicroSDControl( SdkHandle session, PropertyAction bPropAction, unsigned char* enabled );

/// 
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
///	This unlocks the device with the service programming code for activation or configuration.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not an EVDO or EVDO-REVA device
/// 
/// \param[in]  session     Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]  spc         String containing the service programing code.
/// \param[out] success     Receives a value of 0 or 1 depending upon whether or not the service programing code was accepted
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_SetSPC( SdkHandle session, char* spc, unsigned char* success );

///
/// <tt><b>\<CDMA and EVDO devices only\></b></tt>\n
/// This method obtains the mobile identification string of the device. This string comprises of International Mobile Station Identifier (IMSI), Mobile Network Code (MNC) and Mobile Country Code (MCC) values.
/// \n The format of this string is as follows:
/// \n \e MCC-MNC-IMSI where,
///	\n \b MCC - Mobile Country Code
///	\n \b MNC - Mobile Network Code (MNC) for operational IMSI address
///	\n \b IMSI - Operational International Mobile Station Identifier converted to a 32-bit value
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
/// \n \b LR_ERROR_BUFFER_TOO_SMALL if the supplied buffer is not large enough to hold the response
///
/// \param[in]   session        Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]  buffer         A buffer that will receive the value
/// \param[in]   buffer_len     The total length of the buffer that will receive the value
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_GetMobileId( SdkHandle session, char* buffer, unsigned short* buffer_len );

///
///  <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This function retreives the Mobile Equipment Identifier (MEID) for CDMA devices.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_API_NOT_SUPPORTED if MEID is not supported for the device type.
/// \n \b LR_ERROR_BUFFER_TOO_SMALL if the buffer supplied is smaller than the MEID.
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in,out]  buffer          Buffer that will recieve the MEID value
/// \param[in]      buffer_len      Total length of the buffer that will receive the MEID
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_GetMeid( SdkHandle session, char* buffer, unsigned short* buffer_len );


///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This function retreives the esn of the device converted to decimal format.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_OPERATION_NOT_ALLOWED if the device is not a UMTS or HSPA device
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     decEsn          A pointer to a character buffer that will receive the ESN value.
/// \param[in,out]  decEsn_len      A pointer to a short value that contains the length of the input buffer on input, and the length of the ESN on output
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_GetEsnAsDecimal( SdkHandle session, char* decEsn, unsigned short* decEsn_len );


///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This function retreives an entry from the phone book.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_FS an API specific error code is being returned in the errorCode parameter
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_API_NOT_SUPPORTED if the device is not a CDMA device using supported firmware
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      index           Index of the phone book entry
/// \param[out]     buffer          Buffer containing VCard information for the phone book entry
/// \param[in,out]  bufferLength    A pointer to the Length of the input buffer, and the length of the data returned
/// \param[out]     errorCode       A pointer to a 32 bit variable to receive the API specific error code
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_ReadPBEntry( SdkHandle session, unsigned long index, unsigned char *buffer, unsigned long *bufferLength, unsigned long *errorCode );

///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This function writes an entry to the phone book.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_FS an API specific error code is being returned in the errorCode parameter
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_API_NOT_SUPPORTED if the device is not a CDMA device using supported firmware
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      index           Index of the phone book entry
/// \param[in]      buffer          Buffer containing VCard information for the phone book entry
/// \param[in]      bufferLength    Length of the buffer being passed in
/// \param[out]     errorCode       A pointer to a 32 bit variable to receive the API specific error code
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_WritePBEntry( SdkHandle session, unsigned long index, unsigned char *buffer, unsigned long bufferLength, unsigned long *errorCode );

///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This function deletes an entry from the phone book.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_FS an API specific error code is being returned in the errorCode parameter
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_API_NOT_SUPPORTED if the device is not a CDMA device using supported firmware
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[in]      index           Index of the phone book entry
/// \param[out]     errorCode       A pointer to a 32 bit variable to receive the API specific error code
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_ClearPBEntry( SdkHandle session, unsigned long index, unsigned long *errorCode );

///
/// <tt><b>\<CDMA/EVDO devices only\></b></tt>\n
/// This function deletes all entries from the phone book.
///
/// \return
/// type : unsigned short 
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_FS an API specific error code is being returned in the errorCode parameter
/// \n \b LR_ERROR_INVALID_PARAM in the event an invalid parameter is supplied 
/// \n \b LR_ERROR_DEVICE_NOT_AVAILABLE if not currently attached to a device
/// \n \b LR_ERROR_API_NOT_SUPPORTED if the device is not a CDMA device using supported firmware
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     errorCode       A pointer to a 32 bit variable to receive the API specific error code
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_ClearAllPBEntries( SdkHandle session, unsigned long *errorCode );

///
/// This method Gets the availale RSSI values for CDMA based devices.
///
/// \return
/// type : unsigned short
/// \retval A two byte value indicating an error code
/// \n \b LR_ERROR_SUCCESS on success
/// \n \b LR_ERROR_UNKNOWN on failure
/// \n \b LR_ERROR_INVALID_PARAM in event of invalid parameters
///
/// \param[in]      session         Handle to the current sdk session returned from NvtlCommon_CreateSession
/// \param[out]     pRSSI           A pre-alocated pointer to RSSIStruct which will contain the RSSI values for 
///                                 1xRTT and EVDO respectively.
///
///
NOVATEL_API unsigned short NOVATEL_CALL NvtlEvdo_GetRSSI( SdkHandle hClient, RSSIStruct* pRSSI ) ;

//END CDMA FUNCTIONS
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//@}

#ifdef __cplusplus
} //extern "C" {
#endif


#endif //_NW_COMMON_API_H
